from django.contrib.auth.models import User
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required

from django.shortcuts import render_to_response, redirect

from workshop.reg.models import Event
from workshop.reg import forms as reg_forms
from workshop.reg import events as reg_events

from workshop.feedback.models import Feedback

from django.http import HttpResponse

def homepage(request):
    """ see if the user is active.
    If not, only show the re send activation email link.
    else show all the options in homepage.
    """

    user = request.user
    events = Event.objects.all()[:10]

    return render_to_response('index.html', {'user':user, 'events':events})

def user_login(request):
    """ get the user object from e-mail and then check for password.
    """

    user = request.user
    if user.is_authenticated():
        return redirect('/reg')

    if request.method == "POST":
        form = reg_forms.LoginForm(request.POST)
        if form.is_valid():
            email = form.cleaned_data['email']
            password = form.cleaned_data['password']
            username = User.objects.get(email__iexact=email).username

            user = authenticate(username=username, password=password)
            login(request, user)
            return redirect('/reg')
        else:
            return render_to_response('login.html', {'user':user, 'form':form})
    else:
        form = reg_forms.LoginForm()
        return render_to_response('login.html', {'user':user, 'form':form})

def user_logout(request):
    """ simply logout the user and redirect to homepage.
    """

    logout(request)
    return redirect('/reg')

def user_register(request):
    """ take the credentials like name, college and gender here itself.
    """

    if request.method == "POST":
        form = reg_forms.RegisterForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            reg_events.create_user(email=data['email'], 
                                   password=data['password'],
                                   firstname=data['first_name'],
                                   lastname=data['last_name'], 
                                   gender=data['gender'], 
                                   profession=data['profession'], 
                                   affiliated_to=data['affiliated_to'], 
                                   interests=data['interests']
                                  )
            return render_to_response('account_created.html',{'user':user})
        else:
            return render_to_response('register.html', {'user':user, 'form':form})
    else:
        form = reg_forms.RegisterForm()
        return render_to_response('register.html', {'user':user, 'form':form})

def create_event(request):
    """ see if the user is a staff and only then let him do it.
    """

    user = request.user
    if user.is_authenticated() and user.is_staff:
        if request.method ==  "POST":
            form = reg_forms.EventCreateForm(request.POST)
            if form.is_valid():
                data = form.cleaned_data
                new_event = reg_events.create_event(title=data['title'],
                                                    description=data['description'],
                                                    start_date=data['start_date'],
                                                    stop_date=data['stop_date'],
                                                    created_by=user,
                                                   )
                event_url = "/reg/event/view/%s"%(new_event.key)
                return redirect(event_url)
            else:
                return render_to_response('event_create.html', {'user':user, 'form':form})
        else:
            form = reg_forms.EventCreateForm()
            return render_to_response('event_create.html', {'user':user, 'form':form})
    else:
        return redirect('/reg')

def view_event(request, key):
    """ get the event by its key and display it.
    """

    user = request.user
    user_ip = request.META['REMOTE_ADDR']

    try:
        event = Event.objects.get(key__iexact=key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_guest = False if user.is_authenticated() else True
    is_attendee = True if user in event.attendees.all() else False
    is_org = True if user in event.organizers.all() else False

    can_submit_feedback = False
    if not event.feedback_status == "0":
        try:
            event.feedback.get(user_ip__iexact=user_ip, day=event.feedback_status)
        except Feedback.DoesNotExist:
            can_submit_feedback = True 

    context = {'user': user,
               'is_guest': is_guest,
               'event': event,
               'is_attendee': is_attendee,
               'is_org': is_org,
               'can_submit_feedback': can_submit_feedback,
              }
    return render_to_response('view_event.html', context)

def reset_password(request):
    """ check for the existance of e-mail.
    Then call the event.
    """

    user = request.user
    if user.is_authenticated():
        return redirect('/reg')

    if request.method == "POST":
        form = reg_forms.PasswordResetForm(request.POST)
        if form.is_valid():
            email = form.cleaned_data['email']
            user = User.objects.get(email__iexact=email)
            new_password = reg_events.reset_password(user)
            return render_to_response('password_reset.html', {'user':user, 'new_password':new_password})
        else:
            return render_to_response('password_reset.html', {'user':user, 'form':form})
    else:
        form = reg_forms.PasswordResetForm()
        return render_to_response('password_reset.html', {'user':user, 'form':form})

def change_password(request):
    """ check for the password and then set the new password.
    """

    user = request.user
    if not user.is_authenticated():
        return redirect('/reg')

    if request.method == "POST":
        data = request.POST.copy()
        data.__setitem__('username', user.username)
        form = reg_forms.PasswordChangeForm(data)
        if form.is_valid():
            new_password = form.cleaned_data['new_password']
            reg_events.change_password(user, new_password)
            return render_to_response('password_change.html', {'user':user, 'password_changed': True})
        else:
            return render_to_response('password_change.html', {'user':user, 'form':form})
    else:
        form = reg_forms.PasswordChangeForm()
        return render_to_response('password_change.html', {'user':user, 'form':form})

def open_feedback(request, event_key):
    """ see if the event exists.
    then see if feedback is closed.
    then give option for opening the feedback.
    Any feedback can be opened on any day.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_org = True if user in event.organizers.all() else False

    if is_org and event.feedback_status == '0':
        #days = event.stop_date.day() - event.start_date.day() + 1
        days = 2
        if request.method == "POST":
            day = request.POST['day']
            event.feedback_status = day
            event.save()
            return render_to_response('open_feedback.html', {'user':user, 'success': True, 'day':day, 'event':event})
        else:
            return render_to_response('open_feedback.html', {'user':user, 'event': event, 'days': range(1,days+1)})
    else:
        return redirect('/reg')

def close_feedback(request, event_key):
    """ check if the user is org.
    and then check if the feedback is open already.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_org = True if user in event.organizers.all() else False

    if is_org:
        day = event.feedback_status
        event.feedback_status = '0'
        event.save()
        return render_to_response('close_feedback.html', {'user':user, 'event': event, 'day':day})
    else:
        return redirect('/reg')

def open_registration(request, event_key):
    """ simply check for is_org and then set the registration_is_open flag.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_org = True if user in event.organizers.all() else False

    if is_org:
        event.registration_is_open = True
        event.save()
        return render_to_response('reg_open.html', {'user':user, 'event': event})
    else:
        return redirect('/reg')

def close_registration(request, event_key):
    """ simply check for is_org and then unset the registration_is_open flag.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_org = True if user in event.organizers.all() else False

    if is_org:
        event.registration_is_open = False
        event.save()
        return render_to_response('reg_close.html', {'user':user, 'event': event})
    else:
        return redirect('/reg')

def register_for_event(request, event_key):
    """ check if the user is logged in.
    simply add him to the attendees list.
    """

    user = request.user
    if user.is_authenticated():
        try:
            event = Event.objects.get(key__iexact=event_key)
        except Event.DoesNotExist:
            return redirect("/reg")

        event.attendees.add(user)
        return render_to_response("event_register.html", {"user":user, 'event':event})
    else:
        return redirect("/reg")



