#!/usr/bin/python

from django.contrib.auth.models import User
from django.core.mail import send_mail
from django.db import IntegrityError

from ws_app.reg.models import Profile, Event
from ws_app.reg.utils import gen_key

def create_user(email, password, first_name="", last_name="", gender="M", profession="S", affiliated_to="", interests="" ):
    """ create a user with random username and set the password.
    """

    while True:
        try:
            username = gen_key(8)
            user = User.objects.get(username=username)
        except User.DoesNotExist:
            break
    new_user = User.objects.create_user(username, email, password)
    new_user.first_name = first_name
    new_user.last_name = last_name
    new_user.is_active = False
    new_user.save()

    new_profile = Profile(user=new_user)
    new_profile.gender = gender
    new_profile.profession = profession
    new_profile.affiliated_to = affiliated_to
    new_profile.interests = interests

    while True:
        try:
            new_profile.activation_key = gen_key(30)
            new_profile.save()
            return new_user
        except IntegrityError:
            pass

def send_activation(user):
    """ get key from profile and send an email.
    """

    activation_link = "http://fossee.in/workshop/registration/activate/%s"%user.get_profile().activation_key
    subject = "Activate your account"
    message = """
    Dear %s,

    Thank you for registering at fossee.in.
    Your are just a step away from completeing your registration.
    Please click on the link below or open the url in your browser to activate your account.
    %s

    Please mail your queries and complaints to admin@fossee.in.

    If you have not registered at fossee, please ignore this mail.

    Regards,
    FOSSEE Team
    """%(user.get_full_name(), activation_link)

    send_mail(subject, message, "admin@fossee.in", [user.email])

def create_event(title, description, start_date, stop_date, venue, created_by=None):
    """ make an event and save it.
    """

    new_event = Event()
    new_event.title = title
    new_event.description = description
    new_event.start_date = start_date
    new_event.stop_date = stop_date
    new_event.venue = venue
    new_event.save()
    if created_by:
        new_event.organizers.add(created_by)

    while True:
        try:
            new_event.key = gen_key(10)
            new_event.save()
            return new_event
        except IntegrityError:
            pass

def activate_user(user):
    """ mark the is_active flag as true.
    """

    user.is_active = True
    user.save()
    return user

def reset_password(user):
    """ get a key and set it as password.
    for now, print the key.
    later on add the send mail function.
    """

    new_password = gen_key(10)
    user.set_password(new_password)
    user.save()
    return new_password

def change_password(user, new_password):
    """ for now just set the password and be done with it.
    later on, if we want to do something else also then we can add them here.
    """

    user.set_password(new_password)
    user.save()

def update_profile(user, properties):
    """ properties must be a dictionary that contains all the attributes that
    we take while a new user registers.
    """

    profile = user.get_profile()

    user.first_name = properties['first_name']
    user.last_name = properties['last_name']
    user.save()

    profile.gender = properties['gender']
    profile.profession = properties['profession']
    profile.affiliated_to = properties['affiliated_to']
    profile.interests = properties['interests']
    profile.save()


