from django.shortcuts import render_to_response, redirect

from ws_app.reg.models import Event

from ws_app.feedback.models import Feedback, FeedLog
from ws_app.feedback.forms import FeedbackForm
from ws_app.feedback.utils import make_day_report

from django.http import HttpResponse

def submit_feedback(request, event_key):
    """ see if the ip address has already submitted a feedback.
    if not, collect the feedback.
    """

    user = request.user
    if not (user.is_authenticated() and user.is_active):
        raise Http404

    try:
        event = Event.objects.get(key=event_key)
    except Event.DoesNotExist:
        raise Http404

    if event.feedback_status == "0":
        raise Http404
    
    try:
        FeedLog.objects.get(user=user,day=event.feedback_status,event=event)
        return render_to_response('feedback.html', {'user':user, 'submitted':True, 'event':event})
    except FeedLog.DoesNotExist:
        new_feedback = Feedback(event=event)
    
    if request.method == "POST":
        form = FeedbackForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            new_feedback.topics = data['topics']  
            new_feedback.depth = data['depth']
            new_feedback.methodology = data['methodology']
            new_feedback.pace = data['pace']
            new_feedback.applicability = data['applicability']
            new_feedback.problems = data['problems']
            new_feedback.exercises = data['exercises']
            new_feedback.comments = data['comments']
            new_feedback.save()
        
            FeedLog(user=user, event=event, day=event.feedback_status).save()

            return render_to_response('feedback.html', {'user':user, 'submitted':True, 'event':event})
        else:
            return render_to_response('feedback.html',{'user':user, 'form':form, 'event':event})
    else:
        form = FeedbackForm()
        return render_to_response('feedback.html',{'user':user, 'form':form, 'event':event})

def list_feedbacks(request, event_key):
    """ print a list of all the feedbacks collected.
    """

    user = request.user
    
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        raise Http404

    if not user in event.organizers.all():
        raise Http404

    no_of_days = (event.stop_date - event.start_date).days
    days = range(1,no_of_days+2)

    feeds_list = []
    for day in days:
        day_feeds = Feedback.objects.filter(event=event,day=day)
        if day_feeds.count():
            day_list = []
            for feed in day_feeds:
                day_list.append(" ".join((feed.topics, feed.depth, feed.methodology, feed.pace,
                                feed.applicability, feed.problems, feed.exercises, feed.comments)))
            feeds_list.extend(["Day %s"%day, day_list])

    return render_to_response('list_feedbacks.html',{'user':user, 'event':event, 'feeds_list':feeds_list})

def view_report(request, event_key):
    """ compile the report and display it.
    """

    user = request.user
    
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        raise Http404

    if not user in event.organizers.all():
        raise Http404

    no_of_days = (event.stop_date - event.start_date).days
    days = range(1,no_of_days+2)

    workshop_report = []

    for day in days:
        day_num = str(day)
        day_feeds = Feedback.objects.filter(event=event,day=day_num)
        day_report = make_day_report(day_feeds) 
        if day_report:
            workshop_report.extend( [ "Day %s"%day_num, day_report] )

    return render_to_response("show_report.html", {"user":user, "event":event, "workshop_report":workshop_report})
