from datetime import datetime

from django.http import Http404
from django.utils.datastructures import MultiValueDictKeyError

from django.contrib.auth.models import User
from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required

from django.shortcuts import render_to_response, redirect

from workshop.reg.models import Event, Profile
from workshop.reg import forms as reg_forms
from workshop.reg import events as reg_events

from workshop.feedback.models import Feedback

def homepage(request):
    """ see if the user is active.
    If not, only show the re send activation email link.
    else show all the options in homepage.
    """

    user = request.user
    if user.is_authenticated() and user.is_active:
        registered_events = user.event_attendees.all()
    else:
        registered_events = None

    return render_to_response('index.html', {'user':user, 'registered_events':registered_events})

def user_login(request):
    """ get the user object from e-mail and then check for password.
    """

    user = request.user
    if user.is_authenticated():
        return redirect('/reg')

    if request.method == "POST":
        form = reg_forms.LoginForm(request.POST)
        if form.is_valid():
            email = form.cleaned_data['email']
            password = form.cleaned_data['password']
            username = User.objects.get(email__iexact=email).username

            new_user = authenticate(username=username, password=password)
            login(request, new_user)
            return redirect('/reg')
        else:
            return render_to_response('login.html', {'user':user, 'form':form})
    else:
        form = reg_forms.LoginForm()
        return render_to_response('login.html', {'user':user, 'form':form})

def user_logout(request):
    """ simply logout the user and redirect to homepage.
    """

    logout(request)
    return redirect('/reg')

def user_register(request, event_key):
    """ take the credentials like name, college and gender here itself.
    """

    if event_key:
        try:
            event = Event.objects.get(key=event_key)
        except Event.DoesNotExist:
            raise Http404

        if not event.registration_is_open:
            raise Http404
    else:
        event = None

    if request.method == "POST":
        form = reg_forms.RegisterForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            new_user = reg_events.create_user(email=data['email'],
                                              password=data['password'],
                                              first_name=data['first_name'],
                                              last_name=data['last_name'], 
                                              gender=data['gender'], 
                                              profession=data['profession'], 
                                              affiliated_to=data['affiliated_to'], 
                                              interests=data['interests']
                                             )

            reg_events.send_activation(new_user)
            if event:
                event.attendees.add(new_user)
                event.save()

            return redirect('/reg/account_created')
        else:
            return render_to_response('register.html', {'form':form, 'event':event})
    else:
        form = reg_forms.RegisterForm()
        return render_to_response('register.html', {'form':form, 'event':event})

def account_created(request):
    """ simply display a page.
    """
    
    user = request.user
    return render_to_response('account_created.html', {'user':user})

def account_activate(request, activation_key):
    """ see if the key exists.
    see if the corresponding user is inactive.
    """

    user = request.user
    if user.is_authenticated():
        return redirect('/reg')

    try:
        profile = Profile.objects.get(activation_key__iexact=activation_key)
    except Profile.DoesNotExist:
        raise Http404

    new_user = profile.user
    reg_events.activate_user(new_user)
    return render_to_response('account_activated.html', {'user':user})
    
def resend_activation(request):
    """ resend only if user is registered and is inactive.
    """

    user = request.user
    if not user.is_authenticated():
        raise Http404

    try:
        email = request.GET['email']
    except MultiValueDictKeyError:
        raise Http404

    try:
        new_user = User.objects.get(email__iexact=email)
    except User.DoesNotExist:
        raise Http404

    if new_user.is_active:
        return redirect('/reg')
    
    profile = new_user.get_profile()
    activation_key = profile.activation_key
    reg_events.send_activation(new_user)
    
    return render_to_response('sent_activationkey.html', {'user':user})

def create_event(request):
    """ see if the user is a staff and only then let him do it.
    """

    user = request.user
    if user.is_authenticated() and user.is_staff:
        if request.method ==  "POST":
            form = reg_forms.EventCreateForm(request.POST)
            if form.is_valid():
                data = form.cleaned_data
                new_event = reg_events.create_event(title=data['title'],
                                                    description=data['description'],
                                                    start_date=data['start_date'],
                                                    stop_date=data['stop_date'],
                                                    venue=data['venue'],
                                                    created_by=user,
                                                   )
                event_url = "/reg/event/view/%s"%(new_event.key)
                return redirect(event_url)
            else:
                return render_to_response('event_create.html', {'user':user, 'form':form})
        else:
            form = reg_forms.EventCreateForm()
            return render_to_response('event_create.html', {'user':user, 'form':form})
    else:
        raise Http404

def view_event(request, key):
    """ get the event by its key and display it.
    """

    user = request.user
    user_ip = request.META['REMOTE_ADDR']

    try:
        event = Event.objects.get(key__iexact=key)
    except Event.DoesNotExist:
        return redirect("/reg")

    is_guest = False if user.is_authenticated() and user.is_active else True
    is_attendee = True if user in event.attendees.all() else False
    is_org = True if user in event.organizers.all() else False

    can_submit_feedback = False
    if not event.feedback_status == "0":
        try:
            event.feedback.get(user_ip__iexact=user_ip, day=event.feedback_status)
        except Feedback.DoesNotExist:
            can_submit_feedback = True 

    context = {'user': user,
               'is_guest': is_guest,
               'event': event,
               'is_attendee': is_attendee,
               'is_org': is_org,
               'can_submit_feedback': can_submit_feedback,
              }
    return render_to_response('view_event.html', context)

def reset_password(request):
    """ check for the existance of e-mail.
    Then call the event.
    """

    user = request.user
    if user.is_authenticated():
        return redirect('/reg')

    if request.method == "POST":
        form = reg_forms.PasswordResetForm(request.POST)
        if form.is_valid():
            email = form.cleaned_data['email']
            user = User.objects.get(email__iexact=email)
            new_password = reg_events.reset_password(user)
            return render_to_response('password_reset.html', {'user':user, 'new_password':new_password})
        else:
            return render_to_response('password_reset.html', {'user':user, 'form':form})
    else:
        form = reg_forms.PasswordResetForm()
        return render_to_response('password_reset.html', {'user':user, 'form':form})

def change_password(request):
    """ check for the password and then set the new password.
    """

    user = request.user
    if not user.is_authenticated():
        raise Http404

    if request.method == "POST":
        data = request.POST.copy()
        data.__setitem__('username', user.username)
        form = reg_forms.PasswordChangeForm(data)
        if form.is_valid():
            new_password = form.cleaned_data['new_password']
            reg_events.change_password(user, new_password)
            return render_to_response('password_change.html', {'user':user, 'password_changed': True})
        else:
            return render_to_response('password_change.html', {'user':user, 'form':form})
    else:
        form = reg_forms.PasswordChangeForm()
        return render_to_response('password_change.html', {'user':user, 'form':form})

def open_feedback(request, event_key):
    """ see if the event exists.
    then see if feedback is closed.
    then give option for opening the feedback.
    Any feedback can be opened on any day.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        raise Http404

    if user in event.organizers.all() and user.is_staff:
        if event.feedback_status == '0':
            no_of_days = (event.stop_date - event.start_date).days + 1
            if request.method == "POST":
                day = request.POST['day']
                event.feedback_status = day
                event.save()
                return render_to_response('open_feedback.html', {'user':user, 'success': True, 'day':day, 'event':event})
            else:
                return render_to_response('open_feedback.html', {'user':user, 'event': event, 'days': range(1,no_of_days+1)})
        else:
            day = event.feedback_status
            return render_to_response('open_feedback.html', {'user':user, 'success': True, 'day':day, 'event':event})
    else:
        raise Http404

def close_feedback(request, event_key):
    """ check if the user is org.
    and then check if the feedback is open already.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        raise Http404

    if user in event.organizers.all() and user.is_staff:
        day = event.feedback_status
        event.feedback_status = '0'
        event.save()
        return render_to_response('close_feedback.html', {'user':user, 'event': event, 'day':day})
    else:
        raise Http404

def open_registration(request, event_key):
    """ simply check for is_org and then set the registration_is_open flag.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    if user in event.organizers.all() and user.is_staff:
        event.registration_is_open = True
        event.save()
        return render_to_response('reg_open.html', {'user':user, 'event': event})
    else:
        raise Http404

def close_registration(request, event_key):
    """ simply check for is_org and then unset the registration_is_open flag.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect("/reg")

    if user in event.organizers.all() and user.is_staff:
        event.registration_is_open = False
        event.save()
        return render_to_response('reg_close.html', {'user':user, 'event': event})
    else:
        raise Http404

def register_for_event(request, event_key):
    """ check if the user is logged in.
    simply add him to the attendees list.
    """

    user = request.user
    if user.is_authenticated() and user.is_active:
        try:
            event = Event.objects.get(key__iexact=event_key)
        except Event.DoesNotExist:
            return redirect("/reg")

        event.attendees.add(user)
        return render_to_response("event_register.html", {"user":user, 'event':event})
    else:
        return redirect("/reg")

def view_profile(request):
    """ check if user is logged in.
    then show the profile.
    """

    user = request.user
    if not ( user.is_authenticated() and user.is_active ):
        return redirect('/reg')

    user_profile = user.get_profile()
    return render_to_response('view_profile.html', {'user':user, 'user_profile':user_profile})

def edit_profile(request):
    """ check if user is logged in.
    """

    user = request.user
    if not user.is_authenticated():
        return redirect('/reg')

    user_profile = user.get_profile()

    if request.method == "POST":
        form = reg_forms.EditProfileForm(request.POST)
        if form.is_valid():
            reg_events.update_profile(user, form.cleaned_data)
            return redirect('/reg/profile/view')
        else:
            return render_to_response('edit_profile.html', {'user':user, 'form':form})
    else:
        old_info = {'first_name': user.first_name,
                    'last_name': user.last_name,
                    'gender':user_profile.gender,
                    'profession': user_profile.profession,
                    'affiliated_to': user_profile.affiliated_to,
                    'interests': user_profile.interests,
                   }
        form = reg_forms.EditProfileForm(old_info)
        return render_to_response('edit_profile.html', {'user':user, 'form':form})

def list_events(request):
    """ Get all the events including those that are over and list them.
    """

    user = request.user

    today = datetime.now()
    context = {'user':user,
               'upcoming_events': Event.objects.filter(start_date__gt=today),
               'ongoing_events': Event.objects.filter(start_date__lte=today, stop_date__gte=today),
               'previous_events': Event.objects.filter(stop_date__lt=today),
              }

    return render_to_response('list_events.html', context)

def list_attendees(request, event_key):
    """ see if the request user is org.
    Else redirect him to homepage.
    """

    user = request.user
    try:
        event = Event.objects.get(key__iexact=event_key)
    except Event.DoesNotExist:
        return redirect('/reg')

    if not user in event.organizers.all():
        return redirect('/reg')
    
    profile = user.get_profile()
    return render_to_response('list_attendees.html', {'user':user, 'event':event, 'attendees':event.attendees.all()})
