from django.shortcuts import render_to_response
from django.http import Http404

from offline.settings import ADMIN_KEY

from offline.event.models import Event

from offline.feedback.models import Feedback
from offline.feedback.forms import FeedbackForm
from offline.feedback.utils import make_day_report

def submit_feedback(request):

    """ see if the ip address has already submitted a feedback.
    if not, collect the feedback.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404

    if event.feedback_status == "0":
        raise Http404
    else:
        day = event.feedback_status

    ip = request.META['REMOTE_ADDR']

    try:
        Feedback.objects.get(event=event,day=day,user_ip=ip)
        return render_to_response('submit_feedback.html', {'submitted':True, 'event':event})
    except Feedback.DoesNotExist:
        new_feedback = Feedback(event=event,day=day,user_ip=ip)
       
    if request.method == "POST":
        form = FeedbackForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            new_feedback.topics = data['topics']  
            new_feedback.depth = data['depth']
            new_feedback.methodology = data['methodology']
            new_feedback.pace = data['pace']
            new_feedback.applicability = data['applicability']
            new_feedback.problems = data['problems']
            new_feedback.exercises = data['exercises']
            new_feedback.comments = data['comments']
            new_feedback.save()

            return render_to_response('submit_feedback.html', {'submitted':True, 'event':event})
        else:
            return render_to_response('submit_feedback.html',{ 'form':form, 'event':event})
    else:
        form = FeedbackForm()
        return render_to_response('submit_feedback.html',{'form':form, 'event':event})

def open_feedback(request, admin_key):
    """
    Check for key and then exiatance of event.
    """

    if not admin_key == ADMIN_KEY:
        raise Http404

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404

    if event.feedback_status == '0':
        no_of_days = (event.stop_date - event.start_date).days + 1
        if request.method == "POST":
            day = request.POST['day']
            event.feedback_status = day
            event.save()
            return render_to_response('open_feedback.html', {'success': True, 'day':day, 'event':event})
        else:
            return render_to_response('open_feedback.html', {'event': event, 'days': range(1,no_of_days+1)})
    else:
        day = event.feedback_status
        return render_to_response('open_feedback.html', {'success': True, 'day':day, 'event':event})

def close_feedback(request, admin_key):
    """
    simply set feedback status to 0
    """

    if not admin_key == ADMIN_KEY:
        raise Http404

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404

    day = event.feedback_status
    event.feedback_status = '0'
    event.save()
    return render_to_response('close_feedback.html', {'event': event, 'day':day})

def list_feedbacks(request, admin_key):
    """ print a list of all the feedbacks collected.
    """

    if not admin_key == ADMIN_KEY:
        raise Http404

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404


    no_of_days = (event.stop_date - event.start_date).days
    days = range(1,no_of_days+2)

    feeds_list = []
    for day in days:
        day_feeds = Feedback.objects.filter(event=event,day=day)
        if day_feeds.count():
            day_list = []
            for feed in day_feeds:
                day_list.append(" ".join((feed.topics, feed.depth, feed.methodology, feed.pace,
                                feed.applicability, feed.problems, feed.exercises, feed.comments)))
            feeds_list.extend(["Day %s"%day, day_list])

    return render_to_response('list_feedbacks.html',{'event':event, 'feeds_list':feeds_list})

def view_report(request, admin_key):
    """ compile the report and display it.
    """

    if not admin_key == ADMIN_KEY:
        raise Http404

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404

    no_of_days = (event.stop_date - event.start_date).days
    days = range(1,no_of_days+2)

    workshop_report = []

    for day in days:
        day_num = str(day)
        day_feeds = event.feedback.filter(day=day_num)

        day_report = make_day_report(day_feeds) 
        if day_report:
            day_comments = [ feed.comments for feed in day_feeds if feed.comments ]
            day_report.extend(["General comments:",day_comments])
            workshop_report.extend( [ "Day %s"%day_num, day_report] )

    return render_to_response("show_report.html", {"event":event, "workshop_report":workshop_report})
