from django.db.models import Sum
from django.http import Http404
from django.shortcuts import render_to_response, redirect

from offline.settings import ADMIN_KEY

from offline.event.models import Event
from offline.event.forms import EventCreateForm, OpenQuizForm

from offline.quiz.models import Quiz, QuestionBank
from offline.quiz.utils import correct_quiz

num2name = {"11" : "Day 1 Quiz 1",
            "12" : "Day 1 Quiz 2",
            "21" : "Day 2 Quiz 1",
           }

def event_home(request):
    """ The page for people to view.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        raise Http404

    ip = request.META['REMOTE_ADDR']

    can_submit_feedback = True if event.feedback_status != '0' and \
            not event.feedback.filter(day=event.feedback_status,user_ip=ip) else False
    can_take_quiz = True if event.quiz_status != "00" else False

    return render_to_response('home.html', {'event':event, 'can_submit_feedback':can_submit_feedback, 'can_take_quiz':can_take_quiz})

def event_admin(request):
    """ see if the key is correct and then display options.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    return render_to_response('admin.html', {'event':event, 'admin_key':ADMIN_KEY})
    
def event_create(request):
    """ Create an event for which the quiz and feedback should be conducted.
    """

    try:
        event = Event.objects.all()[0]
        return redirect("/event/admin/%s"%admin_key)
    except IndexError:
        new_event = Event()

    if request.method == "POST":
        form = EventCreateForm(request.POST)
        if form.is_valid():
            new_event.title = form.cleaned_data['title']
            new_event.start_date = form.cleaned_data['start_date']
            new_event.stop_date = form.cleaned_data['stop_date']

            new_event.save()
            return redirect('/event/admin/%s'%ADMIN_KEY)
        else:
            return render_to_response('create_event.html',{'form':form})
    else:
        form = EventCreateForm()
        return render_to_response('create_event.html',{'form':form})

def open_quiz(request):
    """ check for admin and then for quiz status.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    if event.quiz_status != "00":
        return redirect('/event/create/%s'%ADMIN_KEY)

    if request.method=="POST":
        form = OpenQuizForm(request.POST)
        if form.is_valid():
            data = form.cleaned_data
            quiz_num = data['quiz_num']
            topics = data['topics']

            for topic in topics:
                QuestionBank.objects.filter(topic=topic).update(quiz_num=quiz_num)

            event.quiz_status = quiz_num
            event.save()
            return render_to_response("open_quiz.html",{"admin_key":ADMIN_KEY, "quiz_name":num2name[quiz_num], 'success':True})
        else:
            return render_to_response("open_quiz.html", {"form":form})
    else:
        form = OpenQuizForm()
        return render_to_response("open_quiz.html", {"form":form})
   
def close_quiz(request):
    """ check for admin and then for quiz status.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    if event.quiz_status == "00":
        return redirect('/event/create/%s'%ADMIN_KEY)
    
    quiz_num = event.quiz_status

    event.quiz_status = "00"
    event.save()

    event_quizzes = event.quiz.filter(quiz_num=quiz_num)
    for quiz in event_quizzes:
        correct_quiz(quiz)

    return render_to_response("close_quiz.html", {"admin_key":ADMIN_KEY, "quiz_name":num2name[quiz_num]})

def list_toppers(request, quiz_num):
    """ check for admin key
    Then check for quiz_num in ['11', ..]
    then give appropriate response.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    if quiz_num not in ["11", "12", "21"]:
        raise Http404

    quizzes = Quiz.objects.filter(event=event,quiz_num=quiz_num)
    quizzes_with_scores = quizzes.annotate(score=Sum('que_answered__is_correct'))
    sorted_quizzes = quizzes_with_scores.order_by("score").reverse()

    return render_to_response("list_toppers.html", {"sorted_quizzes":sorted_quizzes,"admin_key":ADMIN_KEY, "event":event, "quiz_name":num2name[quiz_num]})

def user_dump(request, quiz_key, user_name):
    """ check for admin_key
    then get quiz by quiz_key and user_name.
    then display the dump
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    try:
        quiz = Quiz.objects.get(event=event, key=quiz_key)
    except Quiz.DoesNotExist:
        raise Http404

    if not quiz.user.username == user_name:
        raise Http404

    quiz_name = num2name[quiz.quiz_num]
    if request.method == "POST":
        for ans in quiz.que_answered.all():
            ans.is_correct = True if str(ans.question.id) in request.POST else False
            ans.save()
        return redirect("/quiz/toppers/%s/%s"%(ADMIN_KEY, quiz.quiz_num))
    else:
        return render_to_response("user_dump.html", {"quiz":quiz, "quiz_name":quiz_name, "admin_key":ADMIN_KEY}) 

def que_dump(request, que_id):
    """ check for admin key
    then simply render all answers corresponding to the question.
    """

    try:
        event = Event.objects.all()[0]
    except IndexError:
        return redirect('/event/create/%s'%ADMIN_KEY)

    try:
        question = QuestionBank.objects.get(id=que_id)
    except QuestionBank.DoesNotExist:
        raise Http404

    quiz_num = question.quiz_num
    quiz_name = num2name[quiz_num]

    answers = question.answer.all()

    if request.method == "POST":
        for ans in answers:
            ans.is_correct = True if str(ans.id) in request.POST else False
            ans.save()
        return redirect("/quiz/toppers/%s/%s"%(ADMIN_KEY, quiz_num))
    else:
        return render_to_response("que_dump.html", {"quiz_name":quiz_name, "question":question, "answers":answers})

def list_questions(request):
    """ Display all the questions in a list.
    """

    questions = QuestionBank.objects.all()
    return render_to_response("list_questions.html", {"questions":questions})

