"""This module contains the data model for the project funded by NME
through ICT.
"""


__authors__ = [
  '"Madhusudan.C.S" <madhusudancs@gmail.com>',
]


from django.db import models
from django.contrib.auth.models import User


def sort_dict(d):
  """Function to sort dictionary elements.
  """

  items =  [(v, k) for k, v in d.items()]
  items.sort()
  return [(k, v) for v, k in items]


class Project(models.Model):
  """Model class for NME funded projects.
  """

  LINE_ITEM_CHOICES = {
      "NP": "NPTEL  phase II / III",
      "PG": "PG Classes",
      "UG": "UG Classes",
      "VC": "Video content digitization, conversion,  chunking and dubbing CEC / IGNOU / NCERT / SIET / OTHERS",
      "PE": "Provision of e-books and e-journals free to the learners",
      "SQ": "Standardisation of quality assurance of contents & certification / automation of certification",
      "DS": "Developing suitable pedagogical methods for various classes, intellectual calibers and research in e-learning",
      "DL": "Development of language converter and translation tool kit",
      "DR": "Development and realization of Virtual Reality Laboratories and supporting facilities for e-learning",
      "DC": "Development of Certification & Testing Modules for Virtual Technological Universities & creation of VTU, multi media research and international programmes",
      "ED": "Experimentation and Development of ultra low cost access devices for wider coverage of learners & their field trials",
      "TT": "Talk to a teacher to provide a substitute for coaching for the economically poor students",
      "DH": "Development of software controlled hardware programming for robotics other crucial areas",
      "AD": "Adaptation & deployment of open source simulation packages equivalent to MATLAB, ORCAD etc.",
      "DU": "Development of unified ERP system for Educational Institutions",
      "PT": "Publicity & training of motivators & trainers to ensure full utilization of the systems by institutions & students. Teacher Empowerment 'B'",
      "CC": "Conversion of available content in various regional languages",
      "DV": "Development of Vocational Educational modules and use of haptic devices for education & training",
      }

  STATE_CHOICES = {
      "AN": "Andaman & Nicobar", 
      "DN": "Dadra and Nagar Haveli", 
      "DL": "Delhi", 
      "WB": "West Bengal", 
      "HR": "Haryana", 
      "HP": "Himachal Pradesh", 
      "UP": "Uttar Pradesh", 
      "JH": "Jharkhand", 
      "BR": "Bihar", 
      "JK": "Jammu & Kashmir", 
      "TN": "Tamil Nadu", 
      "LD": "Lakshadweep", 
      "NL": "Nagaland", 
      "PY": "Pondicherry", 
      "TR": "Tripura", 
      "PB": "Punjab", 
      "DD": "Daman & Diu", 
      "RJ": "Rajasthan", 
      "CH": "Chandigarh", 
      "CG": "Chattisgarh", 
      "AP": "Andhra Pradesh", 
      "AS": "Assam", 
      "AR": "Arunachal Pradesh", 
      "GA": "Goa", 
      "GJ": "Gujarat", 
      "KA": "Karnataka", 
      "UA": "Uttarakhand", 
      "ML": "Meghalaya", 
      "MN": "Manipur", 
      "MH": "Maharashtra", 
      "KL": "Kerala", 
      "SK": "Sikkim", 
      "MP": "Madhya Pradesh", 
      "OR": "Orissa", 
      "MZ": "Mizoram"
      }

  DISTRICT_CHOICES = {
      "JHES": "Purba Singhbhum", 
      "TNRA": "Ramanathapuram", 
      "PBAM": "Amritsar", 
      "ORMY": "Mayurbhanj", 
      "UPMG": "Maharajganj", 
      "ORML": "Malkangiri", 
      "MZAI": "Aizawl", 
      "NLKO": "Kohima", 
      "PBFR": "Faridkot", 
      "ARPA": "Papum Pare", 
      "SKES": "East Sikkim", 
      "UAPI": "Pithoragharh", 
      "MHNS": "Nashik", 
      "PBFI": "Firozpur", 
      "KAKD": "Kodagu", 
      "APRA": "Rangareddi", 
      "BRSO": "Sheohar", 
      "UPSV": "Shravasti", 
      "KAKL": "Kolar District", 
      "PYPO": "Puducherry", 
      "UPSU": "Sultanpur", 
      "UPSR": "Sant Ravidas Nagar", 
      "MHRT": "Ratnagiri", 
      "UPSN": "Siddharthnagar", 
      "PBHO": "Hoshiarpur", 
      "UPSJ": "Shahjahanpur", 
      "KAKP": "Koppal District", 
      "UPSI": "Sitapur", 
      "BRSP": "Sheikhpura", 
      "BRSR": "Saran", 
      "BRSU": "Supaul", 
      "BRST": "Sitamarhi", 
      "BRSW": "Siwan", 
      "UPSA": "Saharanpur", 
      "KADH": "Dharwad District", 
      "JHLO": "Lohardaga", 
      "ASSO": "Sonitpur", 
      "KADK": "Dakshina Kannada", 
      "ASSI": "Sibsagar", 
      "KADA": "Davanagere District", 
      "ORJP": "Jajapur (Jajpur)", 
      "ORJS": "Jagatsinghpur", 
      "UPMP": "Mainpuri", 
      "MHBU": "Buldhana", 
      "TNKR": "Karur", 
      "WBME": "Midnapore", 
      "UPUN": "Unnao", 
      "TNKC": "Kanchipuram", 
      "SKSS": "South Sikkim", 
      "MPUM": "Umaria", 
      "JKJA": "Jammu", 
      "MPUJ": "Ujjain", 
      "CGMA": "Mahasamund", 
      "GJPA": "Patan", 
      "UACL": "Chamoli", 
      "PYMA": "Mahe", 
      "GJPO": "Porbandar", 
      "GJPM": "Panchmahal", 
      "RJHA": "Hanumangarh", 
      "WBDD": "Dakshin Dinajpur", 
      "WBDA": "Darjeeling", 
      "HPUN": "Una", 
      "PBFT": "Fatehgarh Sahib", 
      "MHLA": "Latur", 
      "GJVD": "Vadodara", 
      "UPLK": "Lakhimpur Kheri", 
      "JHBO": "Bokaro", 
      "HRHI": "Hissar", 
      "WBJA": "Jalpaiguri", 
      "UPDE": "Deoria", 
      "MPRE": "Rewa", 
      "MPRG": "Rajgarh", 
      "MPRL": "Ratlam", 
      "ASLA": "Lakhimpur", 
      "MPRS": "Raisen", 
      "HPMA": "Mandi", 
      "PBMU": "Mukatsar", 
      "BRPA": "Patna", 
      "BRRO": "Rohtas", 
      "PBMA": "Mansa", 
      "BRPU": "Purnia", 
      "PBMO": "Moga", 
      "RJSR": "Sirohi", 
      "MHAH": "Ahmednagar", 
      "MHAK": "Akola", 
      "MHAM": "Amrawati", 
      "HRSN": "Sonepat", 
      "MHAU": "Aurangabad", 
      "ARTI": "Tirap", 
      "PBBA": "Bathinda", 
      "RJSK": "Sikar", 
      "KAUD": "Udupi District", 
      "RJSM": "Sawai Madhopur", 
      "RJAJ": "Ajmer", 
      "APVZ": "Vizianagaram", 
      "RJAL": "Alwar", 
      "MHWR": "Wardha", 
      "MHWS": "Washim", 
      "APVS": "Vishakhapatnam", 
      "KAHV": "Haveri District", 
      "KAHS": "Hassan District", 
      "MPGW": "Gwalior", 
      "MPGU": "Guna", 
      "JHPK": "Pakur", 
      "JHPL": "Palamu", 
      "UPEW": "Etawah", 
      "UPKU": "Kushinagar", 
      "UPKS": "Kaushambi", 
      "BRSH": "Saharsa", 
      "UPKD": "Kanpur Dehat", 
      "PBPA": "Patiala", 
      "UPKN": "Kanpur Nagar", 
      "BRSM": "Samastipur", 
      "UPVA": "Varanasi", 
      "UPKJ": "Kannauj", 
      "APCH": "Chittoor", 
      "HRRE": "Rewari", 
      "NLZU": "Zunheboto", 
      "BRBU": "Buxar", 
      "HRRO": "Rohtak", 
      "TNCH": "Chennai", 
      "BRBJ": "Bhojpur", 
      "UATG": "Tehri Garhwal", 
      "BRBG": "Bhagalpur", 
      "BRBE": "Begusarai", 
      "BRBA": "Banka", 
      "MHHI": "Hingoli", 
      "UPJP": "Jyotiba Phule Nagar", 
      "GJBR": "Bharuch", 
      "GJBV": "Bhavnagar", 
      "GJBK": "Banaskantha", 
      "TRST": "South Tripura", 
      "GANG": "North Goa", 
      "ORPU": "Puri", 
      "KACK": "Chikballapur District", 
      "UPSO": "Sonbhadra", 
      "HRMW": "Mewat", 
      "RJTO": "Tonk", 
      "NLDI": "Dimapur", 
      "HRMA": "Mahendragarh", 
      "UPSK": "Sant Kabir Nagar", 
      "MZCH": "Champhai", 
      "BRLA": "Lakhisarai", 
      "UPJH": "Jhansi", 
      "ARAJ": "Anjaw", 
      "UAUS": "Udham Singh Nagar", 
      "UPRB": "Rae Bareli", 
      "UAUT": "Uttarkashi", 
      "ASTI": "Tinsukia", 
      "KLKL": "Kollam", 
      "KLKN": "Kannur", 
      "KLKS": "Kasaragod", 
      "KLKT": "Kottayam", 
      "KLKZ": "Kozhikode", 
      "MHBI": "Beed", 
      "MPHA": "Harda", 
      "MHBH": "Bhandara", 
      "MHST": "Satara", 
      "MPHO": "Hoshangabad", 
      "MHSN": "Sangli", 
      "MHSO": "Solapur", 
      "ASBA": "Barpeta", 
      "KASH": "Shimoga District", 
      "MHSI": "Sindhudurg", 
      "ASBO": "Bongaigaon", 
      "MZLU": "Lunglei", 
      "MPCT": "Chhatarpur", 
      "BRKI": "Kishanganj", 
      "BRKH": "Khagaria", 
      "HPSO": "Solan", 
      "KLTV": "Thiruvananthapuram", 
      "BRKM": "Kaimur", 
      "HPSH": "Shimla", 
      "KLTS": "Thrissur", 
      "MNCC": "Churachandpur", 
      "ORBH": "Bhadrak", 
      "MZLA": "Lawngtlai", 
      "BRKT": "Katihar", 
      "ORJH": "Jharsuguda", 
      "MPCN": "Chhindwara", 
      "APCU": "Kadapa", 
      "ORBD": "Boudh (Bauda)", 
      "PYKA": "Karaikal", 
      "UPGZ": "Ghaziabad", 
      "UPGR": "Gorkakhpur", 
      "UPGP": "Ghazipur", 
      "GASG": "South Goa", 
      "JHCH": "Chatra", 
      "MHOS": "Osmanabad", 
      "UPGN": "Gonda", 
      "GJAH": "Ahmedabad", 
      "WBNA": "Nadia", 
      "GJAM": "Amreli District", 
      "GJAN": "Anand", 
      "JKBR": "Baramula", 
      "APGU": "Guntur", 
      "ASMA": "Marigaon", 
      "JKBD": "Badgam", 
      "WBUD": "Uttar Dinajpur", 
      "TNSI": "Sivagangai", 
      "TNSA": "Salem", 
      "MHDH": "Dhule", 
      "PBNS": "Nawan Shehar", 
      "PBGU": "Gurdaspur", 
      "ARWK": "West Kameng", 
      "RJPA": "Pali", 
      "UPAG": "Agra", 
      "WBBI": "Birbhum", 
      "RJPG": "Pratapgarh", 
      "JKPO": "Poonch", 
      "KLPL": "Palakkad", 
      "APPR": "Prakasam", 
      "KLPT": "Pathanamthitta", 
      "TNAY": "Ariyalur", 
      "JKPU": "Pulwama", 
      "RJBW": "Bhilwara", 
      "RJBN": "Banswara", 
      "AREL": "Lohit", 
      "RJBM": "Barmer", 
      "AREK": "East Kameng", 
      "RJBI": "Bikaner", 
      "ORCU": "Cuttack", 
      "RJBU": "Bundi", 
      "RJBR": "Baran", 
      "KLWA": "Wayanad", 
      "RJBP": "Bharatpur", 
      "MHYA": "Yavatmal", 
      "ORAN": "Angul", 
      "UPAU": "Auraiya", 
      "WBMA": "Malda", 
      "MPTI": "Tikamgarh", 
      "GJSN": "Surendranagar", 
      "APNE": "Nellore", 
      "GJSK": "Sabarkantha", 
      "APNI": "Nizamabad", 
      "DLSW": "South West Delhi", 
      "BRGA": "Gaya", 
      "DLSD": "South Delhi", 
      "GJST": "Surat", 
      "UPBD": "Badaun", 
      "ORSO": "Subarnapur (Sonepur)", 
      "JHGI": "Giridih", 
      "JHGO": "Godda", 
      "WBHR": "Howrah", 
      "TNTJ": "Thanjavur", 
      "TNTK": "Thoothukudi", 
      "JHGA": "Garhwa", 
      "TNTI": "Tirunelveli", 
      "TNTL": "Thiruvallur", 
      "ORSA": "Sambalpur", 
      "KLER": "Ernakulam", 
      "TNKK": "Kanyakumari", 
      "TNTP": "Tiruppur", 
      "TNTV": "Tiruvannamalai", 
      "WBMU": "Murshidabad", 
      "MHKO": "Kolhapur", 
      "ORSU": "Sundargarh (Sundergarh)", 
      "UPBG": "Bagpat", 
      "JHGU": "Gumla", 
      "SKNS": "North Sikkim", 
      "RJUD": "Udaipur", 
      "TRDH": "Dhalai", 
      "MLEG": "East Garo Hills", 
      "UPCD": "Chandauli", 
      "HRJH": "Jhajjar", 
      "HRJI": "Jind", 
      "MLSG": "South Garo Hills", 
      "DLCD": "Central Delhi", 
      "UPCT": "Chitrakoot", 
      "MNUK": "Ukhrul", 
      "DLED": "East Delhi", 
      "PBJA": "Jalandhar", 
      "CGRN": "Rajnandgaon", 
      "PYYA": "Yanam", 
      "KABG": "Belgaum District", 
      "KABD": "Bidar District", 
      "KABJ": "Bijapur District", 
      "KABK": "Bagalkot District", 
      "CGRG": "Raigarh", 
      "KABN": "Bangalore Urban district", 
      "KABL": "Bellary District", 
      "KABR": "Bangalore Rural District", 
      "UPPR": "Pratapgarh", 
      "CGRP": "Raipur", 
      "TNTC": "Tiruchirappalli", 
      "UPME": "Meerut", 
      "MLEK": "East Khasi Hills", 
      "DLWD": "West Delhi", 
      "UPMB": "Mau", 
      "UPMO": "Moradabad", 
      "UPMH": "Mahoba", 
      "KATU": "Tumkur District", 
      "MPKA": "Katni", 
      "UPMT": "Mathura", 
      "UPMU": "Muzaffarnagar", 
      "TNMA": "Madurai", 
      "JKLE": "Leh", 
      "APMA": "Mahbubnagar", 
      "ASCA": "Cachar", 
      "APME": "Medak", 
      "MHRG": "Raigad", 
      "MPBP": "Bhopal", 
      "MPBR": "Barwani", 
      "MNBI": "Bishnupur", 
      "GJVL": "Valsad", 
      "NLPH": "Phek", 
      "RJNA": "Nagaur", 
      "BRDA": "Darbhanga", 
      "MPBE": "Betul", 
      "MPBD": "Bhind", 
      "PBLU": "Ludhiana", 
      "MPBL": "Balaghat", 
      "MZKO": "Kolasib", 
      "APNA": "Nalgonda", 
      "HPBI": "Bilaspur", 
      "UPJU": "Jaunpur District", 
      "GJDG": "The Dangs", 
      "GJDA": "Dahod", 
      "WBHG": "Hooghly", 
      "NLWO": "Wokha", 
      "MPPA": "Panna", 
      "UPJL": "Jalaun", 
      "ASJO": "Jorhat", 
      "WBKO": "Kolkata", 
      "HPKU": "Kulu", 
      "HRSI": "Sirsa", 
      "MLJH": "Jaintia Hills", 
      "HPKI": "Kinnaur", 
      "UADD": "Dehradun", 
      "UACP": "Champawat", 
      "HPKA": "Kangra", 
      "TNPU": "Pudukkottai", 
      "JHKO": "Koderma", 
      "UPRA": "Rampur", 
      "TNPE": "Perambalur", 
      "HRAM": "Ambala", 
      "MHGO": "Gondiya", 
      "KLAL": "Alappuzha", 
      "MHGA": "Gadchiroli", 
      "KAMA": "Mandya District", 
      "BRMP": "Madhepura", 
      "BRMZ": "Muzaffarpur", 
      "BRMG": "Munger", 
      "JKSR": "Samba", 
      "BRMB": "Madhubani", 
      "KAMY": "Mysore District", 
      "RJGA": "Ganganagar", 
      "BRGO": "Gopalganj", 
      "ORDH": "Dhenkanal", 
      "MPEN": "Khandwa (East Nimar)", 
      "JHRA": "Ranchi", 
      "JHRM": "Ramgarh District", 
      "BRVA": "Vaishali", 
      "HPLS": "Lahaul and Spiti", 
      "GJKA": "Kutch", 
      "UARP": "Rudraprayag", 
      "GJKH": "Kheda", 
      "MPSI": "Sidhi", 
      "MPWN": "Khargone (West Nimar)", 
      "PBRU": "Rupnagar", 
      "MNSE": "Senapati", 
      "APAN": "Anantapur", 
      "NLTU": "Tuensang", 
      "CGJA": "Jashpur", 
      "ASGG": "Golaghat", 
      "CGJC": "Janjgir-Champa", 
      "APAD": "Adilabad", 
      "RJJW": "Jhalawar", 
      "UPMI": "Mirzapur", 
      "RJJS": "Jaisalmer", 
      "RJJP": "Jaipur", 
      "ASGP": "Goalpara", 
      "RJJL": "Jalore", 
      "RJJJ": "Juhnjhunun", 
      "HPHA": "Hamirpur", 
      "MPSR": "Sehore", 
      "TNTH": "Theni", 
      "MLWG": "West Garo Hills", 
      "CGDH": "Dhamtari", 
      "MHND": "Nanded", 
      "MHNG": "Nagpur", 
      "MLWK": "West Khasi Hills", 
      "CGDA": "Dantewada", 
      "HRFT": "Fatehabad", 
      "CGDU": "Durg", 
      "HRFR": "Faridabad", 
      "TNTR": "Thiruvarur", 
      "UPFI": "Firozabad", 
      "JKAN": "Anantnag", 
      "UPFT": "Fatehpur", 
      "ASNL": "Nalbari", 
      "APKA": "Karimnagar", 
      "HRPP": "Panipat", 
      "UPFR": "Farrukhabad", 
      "ASNG": "Nagaon", 
      "ASNC": "North Cachar Hills", 
      "UPFZ": "Faizabad", 
      "MNTH": "Thoubal", 
      "WBPN": "North 24 Parganas", 
      "TRWT": "West Tripura", 
      "MNTA": "Tamenglong", 
      "PBKA": "Kapurthala", 
      "RJDA": "Dausa", 
      "WBPS": "South 24 Parganas", 
      "UPPI": "Pilibhit", 
      "KLMA": "Malappuram", 
      "ARCH": "Changlang", 
      "WBPU": "Purulia", 
      "MHCH": "Chandrapur", 
      "KACJ": "Chamarajnagar District", 
      "ORKN": "Kandhamal", 
      "ORKO": "Koraput", 
      "ORKL": "Kalahandi", 
      "ORKJ": "Kendujhar (Keonjhar)", 
      "ORKH": "Khordha", 
      "NLMK": "Mokokchung", 
      "ORKP": "Kendrapara", 
      "NLMN": "Mon", 
      "KACT": "Chitradurga District", 
      "CGSU": "Surguja", 
      "MPNE": "Neemuch", 
      "TNNM": "Namakkal", 
      "MPNA": "Narsinghpur", 
      "TNNI": "The Nilgiris", 
      "TNNG": "Nagapattinam", 
      "APHY": "Hyderabad", 
      "UANA": "Nainital", 
      "APSR": "Srikakulam", 
      "RJCR": "Churu", 
      "RJCT": "Chittorgarh", 
      "MZMA": "Mamit", 
      "MHMC": "Mumbai City", 
      "UPET": "Etah", 
      "GJGA": "Gandhinagar", 
      "MHMU": "Mumbai suburban", 
      "WBKB": "Cooch Behar", 
      "HPCH": "Chamba", 
      "MPSG": "Sagar", 
      "HRPK": "Panchkula", 
      "ASKR": "Karimganj", 
      "MPSO": "Seoni", 
      "GJNV": "Navsari", 
      "MPSJ": "Shajapur", 
      "MPSH": "Shahdol", 
      "GJNR": "Narmada", 
      "MPSV": "Shivpuri", 
      "MPST": "Satna", 
      "ASKA": "Karbi Anglong", 
      "MPSP": "Sheopur", 
      "APEG": "East Godavari", 
      "HRPW": "Palwal", 
      "JKDO": "Doda", 
      "ASKK": "Kokrajhar", 
      "JHDU": "Dumka", 
      "ORRA": "Rayagada", 
      "JHDE": "Deoghar", 
      "MHJG": "Jalgaon", 
      "HRBH": "Bhiwani", 
      "TNDH": "Dharmapuri", 
      "JHDH": "Dhanbad", 
      "MHJN": "Jalna", 
      "UPBI": "Bijnor", 
      "UPBH": "Bahraich", 
      "UPBL": "Ballia", 
      "UPBN": "Banda District", 
      "HRKR": "Karnal", 
      "UPBB": "Barabanki", 
      "MZSE": "Serchhip", 
      "HRKU": "Kurukshetra", 
      "HRKT": "Kaithal", 
      "MNCD": "Chandel", 
      "MZSA": "Saiha", 
      "UPBP": "Balrampur", 
      "UPBS": "Basti", 
      "UPBR": "Bareilly", 
      "UPBU": "Bulandshahr", 
      "MLRB": "Ri-Bhoi", 
      "JKRA": "Rajauri", 
      "ORBR": "Bargarh (Baragarh)", 
      "BRNW": "Nawada", 
      "MHTH": "Thane", 
      "HPSI": "Sirmaur", 
      "RJDU": "Dungapur", 
      "APWA": "Warangal", 
      "ORBW": "Baleswar (Balasore)", 
      "TNER": "Erode", 
      "BRNL": "Nalanda", 
      "MPDE": "Dewas", 
      "KAGU": "Gulbarga District", 
      "TRNT": "North Tripura", 
      "KLID": "Idukki", 
      "MPDI": "Dindori", 
      "MPDH": "Dhar", 
      "KAGA": "Gadag District", 
      "MPDT": "Datia", 
      "JHSA": "Sahibganj", 
      "ORGP": "Gajapati", 
      "UAPG": "Pauri Garhwal", 
      "ORBL": "Bolangir (Balangir)", 
      "MPJH": "Jhabua", 
      "UPLA": "Lalitpur", 
      "MPJA": "Jabalpur", 
      "JKKR": "Kargil", 
      "JKKT": "Kathua", 
      "JKKU": "Kupwara", 
      "WBBN": "Bankura", 
      "KAUK": "Uttara Kannada", 
      "ASDI": "Dibrugarh", 
      "UPLU": "Lucknow", 
      "WBBR": "Bardhaman", 
      "ASDM": "Dhemaji", 
      "ASDA": "Darrang", 
      "ASDB": "Dhubri", 
      "HRYN": "Yamuna Nagar", 
      "BREC": "Purba Champaran", 
      "UABA": "Bageshwar", 
      "TNCU": "Cuddalore", 
      "MPMS": "Mandsaur", 
      "MPML": "Mandla", 
      "GJRA": "Rajkot", 
      "MPMO": "Morena", 
      "TNCO": "Coimbatore", 
      "GJJU": "Junagadh", 
      "TNVE": "Vellore", 
      "TNVL": "Villupuram", 
      "UPHR": "Hardoi", 
      "ORDE": "Debagarh (Deogarh)", 
      "HRGU": "Gurgaon", 
      "RJJO": "Jodhpur", 
      "UPAH": "Allahabad", 
      "UPAL": "Aligarh", 
      "UPAN": "Ambedkar Nagar", 
      "CGBI": "Bilaspur", 
      "APWG": "West Godavari", 
      "UPAZ": "Azamgarh", 
      "CGBA": "Bastar", 
      "ORNY": "Nayagarh", 
      "BRWC": "Pashchim Champaran", 
      "ORNU": "Nuapada", 
      "MNWI": "Imphal West", 
      "UAAL": "Almora", 
      "JHHA": "Hazaribagh", 
      "ORNB": "Nabarangpur", 
      "ARUD": "Dibang Valley", 
      "ARUB": "Upper Subansiri", 
      "RJRA": "Rajsamand", 
      "APKR": "Krishna", 
      "MHPU": "Pune", 
      "APKU": "Kurnool", 
      "MPIN": "Indore", 
      "ASHA": "Hailakandi", 
      "UPGB": "Gautam Buddha Nagar", 
      "MHPA": "Parbhani", 
      "KARA": "Raichur District", 
      "SKWS": "West Sikkim", 
      "APKH": "Khammam", 
      "KARM": "Ramanagara District", 
      "DLNO": "North Delhi", 
      "BRJE": "Jehanabad", 
      "BRJA": "Jamui", 
      "JHWS": "Pashchim Singhbhum", 
      "DLND": "New Delhi", 
      "DLNE": "North East Delhi", 
      "DLNW": "North West Delhi", 
      "RJDH": "Dholpur", 
      "TNDI": "Dindigul", 
      "JKUD": "Udhampur", 
      "GJJA": "Jamnagar", 
      "UPHM": "Hamirpur", 
      "PBSA": "Sangrur", 
      "MPVI": "Vidisha", 
      "ORGN": "Ganjam", 
      "MHNB": "Nandurbar", 
      "UPHT": "Mahamaya Nagar", 
      "BRAR": "Araria", 
      "JKBP": "Bandipore", 
      "CGKB": "Korba", 
      "BRAU": "Aurangabad", 
      "CGKK": "Kanker", 
      "CGKJ": "Koriya", 
      "CGKW": "Kawardha", 
      "UAHA": "Haridwar", 
      "GJMA": "Mehsana", 
      "RJKO": "Kota", 
      "MNEI": "Imphal East", 
      "RJKA": "Karauli", 
      "ARLB": "Lower Subansiri", 
      "MPDM": "Damoh"
      }

  # Field containing the Line Item to which the project belongs to.
  line_item = models.CharField(max_length=256,
                               choices=sort_dict(LINE_ITEM_CHOICES))
  line_item.help_text = 'Select from one of the Line Items.'

  # Field containing the name of the institution working on the
  # project.
  institution = models.CharField(max_length=256)
  institution.help_text = 'Give the full name of your institution.'

  # Field containing the state to which the institution belongs to.
  state = models.CharField(
      max_length=256,
      choices=sorted(STATE_CHOICES.items(), key=lambda (k,v): (v,k)))
  state.help_text = 'Select the state in which this project is taken up.'

  # Field containing the district to which the institution belongs
  # to in the state of India.
  district = models.CharField(max_length=256,
                              choices=sort_dict(DISTRICT_CHOICES))
  district.help_text = ('First select the state before selecting the district. '
      'Select the district of the state where this project is taken up.')

  mobile_num = models.CharField(max_length=20)
  mobile_num.help_text = 'Enter your mobile number.'

  fax_num = models.CharField(max_length=20, null=True, blank=True)
  fax_num.help_text = 'Enter your fax number with the code.'

  # Field containing the autogenerated MICR code for the project.
  micr_code = models.CharField(max_length=15, unique=True)

  # Field containing the status of the project.
  # status of the project can be one among the following
  # New, Revised, Funded, Pilot, DPE
  status = models.CharField(max_length=256,
                            choices=[('new', 'New'), ('pilot', 'Pilot'),
                                     ('invalid', 'Invalid')])

  last_updated_on = models.DateTimeField(auto_now=True)

  @classmethod
  def getLineItem(cls, code):
    """Get the State name from its code.
    """

    return cls.LINE_ITEM_CHOICES[code]

  @classmethod
  def getLineItemCode(cls, name):
    """Get the Line Item code from its name.
    """

    for ln_code in cls.LINE_ITEM_CHOICES:
      if cls.LINE_ITEM_CHOICES[ln_code] == name:
        return ln_code

    return None

  @classmethod
  def getState(cls, code):
    """Get the State code from its name.
    """

    return cls.STATE_CHOICES[code]

  @classmethod
  def getStateCode(cls, name):
    """Get the State code from its name.
    """

    for st_code in cls.STATE_CHOICES:
      if cls.STATE_CHOICES[st_code] == name:
        return st_code

    return None

  @classmethod
  def getDistrict(cls, code):
    """Get the District name from its code.
    """

    return cls.DISTRICT_CHOICES[code]

  @classmethod
  def getDistrictCode(cls, name):
    """Get the District code from its name.
    """

    for dt_code in cls.DISTRICT_CHOICES:
      if cls.DISTRICT_CHOICES[dt_code] == name:
        return dt_code

    return None

class Proposal(models.Model):
  """Model class for the project's proposal.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)
 
  #: Field containing the Line Item to which the project belongs to.
  document = models.FileField(upload_to='proposals/%Y/%m/%d')
  document.help_text = 'Select the document path from your local file system.'

  #: Field containing the date on which the document was submitted
  submitted_on = models.DateTimeField(auto_now_add=True)

  #: Field containing the reference to the user who submitted the proposal.
  submitted_by = models.ForeignKey(User, null=True)

  #: Field containing the revision number of the proposal belonging to
  #: the Project.
  rev_num = models.PositiveIntegerField()


class Timeline(models.Model):
  """Model class for the project's timeline.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the date and time of submission of proposal.
  submitted = models.DateTimeField()

  #: Field containing the last date and time of review of proposal.
  reviewed = models.DateTimeField()

  #: Field containing the date and time of amount paid for the project.
  amount_paid = models.DateTimeField()

  #: Field containing the date and time of presentation of the project.
  presentation = models.DateTimeField()

  #: Field containing the date and time of monitoring of the project.
  monitoring = models.DateTimeField()


class Fund(models.Model):
  """Model class for the project's funds.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the amount sanctioned as funds for the project.
  sanctioned = models.FloatField()

  #: Field containing the expenses for the sanctioned fund for
  #: the project.
  expenses = models.FloatField()

  #: Field containing the date and time on which the funds were
  #: sanctioned for the project.
  sanctioned_on = models.DateTimeField(auto_now_add=True)


class Review(models.Model):
  """Model class for the project's proposal's review.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the comment entered along with the review.
  comment = models.TextField()
  comment.help_text = "Enter your review comment about this proposal."

  #: Field representing the reference to the person who
  #: did the review.
  reviewer = models.ForeignKey(User, null=True)

  #: Field containing the date and time of review of the proposal.
  reviewed_on = models.DateTimeField(auto_now=True)

  #: Field containing the review value for this attribute.
  attribute1 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])
  
  attribute2 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute3 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute4 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute5 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute6 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute7 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute8 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute9 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])
