"""This module contains the data model for the project funded by NME
through ICT.
"""


__authors__ = [
  '"Madhusudan.C.S" <madhusudancs@gmail.com>',
]


from django.db import models
from django.contrib.auth.models import User


class Project(models.Model):
  """Model class for NME funded projects.
  """

  LINE_ITEM_CHOICES = [('ME', 'Mechanical'), 
                       ('CE', 'Chemical'),
                       ('EE', 'Electrical'),
                       ('AE', 'Aero Space'),
                       ('CS', 'Computer Science'),
                       ('IT', 'Information Technology'),
                      ]

  STATE_CHOICES = [('MH', 'Maharashtra'), 
                   ('KA', 'Karnataka'),
                   ('KL', 'Kerala'),
                   ('TN', 'Tamil Nadu'),
                   ('AP', 'Andra Pradesh'),
                  ]

  DISTRICT_CHOICES = [('AD', 'Adilabad'),
                      ('RT', 'Ratnagiri'),
                      ('MU', 'Mumbai suburban'),
                      ('PU', 'Pune'),
                      ('PL', 'Palakkad'),
                      ('BN', 'Bangalore Urban district'),
                      ('CK', 'Chikmagalur District'),
                     ]

  # Field containing the Line Item to which the project belongs to.
  line_item = models.CharField(max_length=256,
                               choices=LINE_ITEM_CHOICES)

  # Field containing the name of the institution working on the
  # project.
  institution = models.CharField(max_length=256)

  # Field containing the state to which the institution belongs to.
  state = models.CharField(max_length=256,
                           choices=STATE_CHOICES)

  # Field containing the district to which the institution belongs
  # to in the state of India.
  district = models.CharField(max_length=256,
                              choices=DISTRICT_CHOICES)

  # Field containing the autogenerated MICR code for the project.
  micr_code = models.CharField(max_length=15, unique=True)

  # Field containing the status of the project.
  # status of the project can be one among the following
  # New, Revised, Funded, Pilot, DPE
  status = models.CharField(max_length=256,
                            choices=[('new', 'New'), ('pilot', 'Pilot'),
                                     ('invalid', 'Invalid')])

  @classmethod
  def getLineItem(cls, code):
    """Get the State name from its code.
    """

    line_item_dict = dict(cls.LINE_ITEM_CHOICES)
    return line_item_dict[code]

  @classmethod
  def getLineItemCode(cls, name):
    """Get the Line Item code from its name.
    """

    for ln_code, ln_name in cls.LINE_ITEM_CHOICES:
      if ln_name == name:
        return ln_code

    return None

  @classmethod
  def getState(cls, code):
    """Get the State code from its name.
    """

    state_dict = dict(cls.STATE_CHOICES)
    return state_dict[code]

  @classmethod
  def getStateCode(cls, name):
    """Get the State code from its name.
    """

    for st_code, st_name in cls.STATE_CHOICES:
      if st_name == name:
        return st_code

    return None

  @classmethod
  def getDistrict(cls, code):
    """Get the District name from its code.
    """

    district_dict = dict(cls.DISTRICT_CHOICES)
    return district_dict[code]

  @classmethod
  def getDistrictCode(cls, name):
    """Get the District code from its name.
    """

    for dt_code, dt_name in cls.DISTRICT_CHOICES:
      if dt_name == name:
        return dt_code

    return None

class Proposal(models.Model):
  """Model class for the project's proposal.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)
 
  #: Field containing the Line Item to which the project belongs to.
  document = models.FileField(upload_to='proposals/%Y/%m/%d')

  #: Field containing the date on which the document was submitted
  submitted_on = models.DateTimeField(auto_now_add=True)

  #: Field containing the reference to the user who submitted the proposal.
  submitted_by = models.ForeignKey(User, null=True)

  #: Field containing the revision number of the proposal belonging to
  #: the Project.
  rev_num = models.PositiveIntegerField()


class Timeline(models.Model):
  """Model class for the project's timeline.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the date and time of submission of proposal.
  submitted = models.DateTimeField()

  #: Field containing the last date and time of review of proposal.
  reviewed = models.DateTimeField()

  #: Field containing the date and time of amount paid for the project.
  amount_paid = models.DateTimeField()

  #: Field containing the date and time of presentation of the project.
  presentation = models.DateTimeField()

  #: Field containing the date and time of monitoring of the project.
  monitoring = models.DateTimeField()


class Fund(models.Model):
  """Model class for the project's funds.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the amount sanctioned as funds for the project.
  sanctioned = models.FloatField()

  #: Field containing the expenses for the sanctioned fund for
  #: the project.
  expenses = models.FloatField()

  #: Field containing the date and time on which the funds were
  #: sanctioned for the project.
  sanctioned_on = models.DateTimeField(auto_now_add=True)


class Review(models.Model):
  """Model class for the project's proposal's review.
  """

  #: Field representing the relation to the corresponding project.
  project = models.ForeignKey(Project)

  #: Field containing the comment entered along with the review.
  comment = models.TextField()

  #: Field representing the reference to the person who
  #: did the review.
  reviewer = models.ForeignKey(User, null=True)

  #: Field containing the date and time of review of the proposal.
  reviewed_on = models.DateTimeField(auto_now_add=True)

  #: Field containing the review value for this attribute.
  attribute1 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])
  
  attribute2 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute3 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute4 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute5 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute6 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute7 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute8 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])

  attribute9 = models.PositiveSmallIntegerField(
      choices=[(1, 1), (2, 2), (3, 3), (4, 4), (5, 5)])
