"""This module contains the views for the project's proposal 
funded by NME through ICT.
"""


__authors__ = [
  '"Madhusudan.C.S" <madhusudancs@gmail.com>',
]


import time

from django.shortcuts import render_to_response, get_object_or_404

from projrev.models import Project
from projrev.models import Proposal
from projrev.views.helpers import forms as projrev_forms


def submit(request):
  """View for proposal submission.
  """

  if request.method == 'POST':
    return submitPost(request)
  else:
    return submitGet(request)

def submitPost(request):
  """Handles POST request for the submitted proposal form. 
  """

  prop_form = projrev_forms.ProposalForm(request.POST, request.FILES)

  project = None
  proposal = None

  if prop_form.is_valid():
    cleaned_data = prop_form.cleaned_data

    # Generate MICR code
    cleaned_data['micr_code'] = '%s%s%s%d' % (
        cleaned_data['state'], cleaned_data['district'],
        cleaned_data['line_item'], 
        int(time.time() * 1000) % 1000000000)

    cleaned_data['line_item'] = Project.getLineItem(cleaned_data['line_item'])
    cleaned_data['state'] = Project.getState(cleaned_data['state'])
    cleaned_data['district'] = Project.getDistrict(cleaned_data['district'])

    # If the form is valid create a new project or update the project
    # if it already exists from the form.
    project = prop_form.save()

    project.status = 'new'

    project.save()

    # Create a proposal for the project.
    proposal = project.proposal_set.create(
        document=prop_form.cleaned_data['document'], rev_num = 0)

    proposal.save()

  return submitGet(request, project, proposal)

def submitGet(request, project=None, proposal=None):
  """Handles GET request for the submission of proposal form.
  """

  context = {}

  if proposal:
    context['document'] = proposal.document

  if not project:
    prop_form = projrev_forms.ProposalForm()
  else:
    prop_form = projrev_forms.ProposalForm(instance=project)

  context['form'] = prop_form

  template = 'projrev/proposal/submit.html'

  return render_to_response(template, context)

def review(request, micr_code=None):
  """
  """

  if request.method == 'POST':
    return reviewPost(request, micr_code)
  else:
    return reviewGet(request, micr_code)

def reviewPost(request, micr_code=None):
  """
  """

  rev_form = projrev_forms.ReviewForm(request.POST)

  if rev_form.is_valid():
    cleaned_data = rev_form.cleaned_data

    cleaned_data['project'] = Project.objects.get(micr_code=micr_code)

    # If the form is valid create a new project or update the project
    # if it already exists from the form.
    review = rev_form.save()

  return reviewGet(request, micr_code, rev_form)

def reviewGet(request, micr_code=None, rev_form=None):
  """
  """

  if not micr_code:
    template = 'projrev/proposal/list.html'
    context = {
        'projects': Project.objects.all(),
        'row_url': '/proposal/review/',
        }

    return render_to_response(template, context)

  if not rev_form:
    rev_form = projrev_forms.ReviewForm()

  proposal_path = str(Project.objects.get(
          micr_code=micr_code).proposal_set.all()[0].document)

  proposal_name = proposal_path.split('/')[-1]

  context = {
      'form': rev_form,
      'project': Project.objects.get(micr_code=micr_code),
      'proposal_path': proposal_path,
      'proposal_name': proposal_name,
      }

  template = 'projrev/proposal/review.html'

  return render_to_response(template, context)

def rank(request, micr_code=None):
  """
  """

  if request.method == 'POST':
    return rankPost(request, micr_code)
  else:
    return rankGet(request, micr_code)

def rankPost(request, micr_code=None):
  """
  """

  return rankGet(request, micr_code)

def rankGet(request, micr_code=None):
  """
  """

  if not micr_code:
    template = 'projrev/proposal/list.html'
    context = {
        'projects': Project.objects.all(),
        'row_url': '/proposal/rank/',
        }

    return render_to_response(template, context)

  projects = Project.objects.get(micr_code=micr_code)

  proposal_path = str(projects.proposal_set.all()[0].document)

  proposal_name = proposal_path.split('/')[-1]

  reviews = projects.review_set.all()

  review_score = [0] * 9
  for review in reviews:
    review_score[0] += review.attribute1
    review_score[1] += review.attribute2
    review_score[2] += review.attribute3
    review_score[3] += review.attribute4
    review_score[4] += review.attribute5
    review_score[5] += review.attribute6
    review_score[6] += review.attribute7
    review_score[7] += review.attribute8
    review_score[8] += review.attribute9

  total_score = sum(review_score)

  context = {
      'project': projects,
      'proposal_path': proposal_path,
      'proposal_name': proposal_name,
      'review_score': review_score,
      'total_score': total_score,
      }

  template = 'projrev/proposal/rank.html'

  return render_to_response(template, context)
