#!/usr/bin/python
# Contains all the polynomial functions

import pylab as pl

def polsize(Q, degQ):
    """ Determines dimensions of a polynomial matrix. """
    rQ, cQ = pl.atleast_2d(Q).shape
    cQ = cQ/float(degQ+1)
    if abs(round(cQ)-cQ) > 1e-6:
        print "Degree of input inconsistent with number of columns"
        return
    else:
        cQ = int(round(cQ))
    return rQ, cQ

def polmul(A, degA, B, degB):
    A = pl.atleast_2d(A)
    B = pl.atleast_2d(B)
    rA, cA = polsize(A, degA)
    rB, cB = polsize(B, degB)
    if cA != rB:
        print "polmul: Inconsistent dimensions of input matrices"
        return
    degC = degA + degB
    C = []
    for k in range(0, degC+1):
        mi = 0
        if k-degB > mi:
            mi = k-degB
        ma = degA
        if k < ma:
            ma = k
        Ck = pl.zeros((rA,cB))
        for i in range(mi, ma+1):
            Ck = Ck + pl.dot(A[..., i*cA:(i+1)*cA], B[..., (k-i)*cB:(k-i+1)*cB])
        Ck = pl.squeeze(Ck)
        C = pl.hstack((C, Ck))
    return C, degC

def poladd(A, degA, B, degB):
    A = pl.atleast_2d(A)
    B = pl.atleast_2d(B)
    rA, cA = polsize(A, degA)
    rB, cB = polsize(B, degB)
    if cA != rB:
        print "polmul: Inconsistent dimensions of input matrices"
        return
    degC = max(degA, degB)

    if degC >= degA:
        A = pl.hstack((A, pl.zeros((rA,(degC-degA)*cA))))

    if degC >= degB:
        B = pl.hstack((B, pl.zeros((rB,(degC-degB)*cB))))

    C = A + B
    return C, degC

def polsplit2(fac, a=1-1e-5):
    fac = pl.atleast_1d(fac)
    if a>1:
        print "good polynomial also is unstable"
        return
    roots = pl.roots(fac)

    # extract good and bad roots
    badindex = pl.find(pl.absolute(roots)>=a-1.0e-5)
    badpoly = pl.poly(roots[badindex])
    goodindex = pl.find(pl.absolute(roots)<a-1.0e-5)
    goodpoly = pl.poly(roots[goodindex])
    # scale by equating the largest terms
    index = pl.absolute(fac).argmax()
    goodbad = pl.convolve(goodpoly, badpoly)
    factor = fac[index]/goodbad[index]
    goodpoly = goodpoly * factor
    badpoly = pl.atleast_1d(badpoly)
    goodpoly = pl.atleast_1d(goodpoly)
    return goodpoly, badpoly

def polsplit3(fac, a=1):
    fac = pl.atleast_1d(fac)
    if a>1:
        print "good polynomial also is unstable"
        return
    roots = pl.roots(fac)

    # extract good and bad roots
    badindex = pl.find((pl.absolute(roots)>=a-1.0e-5) + (pl.real(roots)<-0.05))
    badpoly = pl.poly(roots[badindex])
    goodindex = pl.find((pl.absolute(roots)<a-1.0e-5) * (pl.real(roots)>=-0.05))
    goodpoly = pl.poly(roots[goodindex])
    # scale by equating the largest terms
    index = pl.absolute(fac).argmax()
    goodbad = pl.convolve(goodpoly, badpoly)
    factor = fac[index]/goodbad[index]
    goodpoly = goodpoly * factor
    badpoly = pl.atleast_1d(badpoly)
    goodpoly = pl.atleast_1d(goodpoly)
    return goodpoly, badpoly

def putin(A, degA, B, degB, i, j):
    from clcoef import clcoef
    A = pl.atleast_2d(A)
    B = pl.atleast_2d(B)
    rA, cA = polsize(A,degA)
    if degB > degA:
        A = pl.hstack((A, pl.zeros((rA,(degB-degA)*cA))))
    degA = degB

    for k in range(degB+1):
        A[i,(k*cA)+j] = B[0,k]

    if degA > degB:
        for k in range(degB+1,degA+1):
            A[i,(k*cA)+j] = 0
        A, degA = clcoef(A,degA)
    return A, degA


def ext(A, degA, k, l):
    from clcoef import clcoef
    rA, cA = polsize(A, degA)
    degB = degA
    B = pl.zeros((1, degB+1))
    for m in range(degB+1):
        B[0, m] = A[k, (m*cA)+l]
    B,degB = clcoef(B, degB)
    return B, degB


def transp(Q, degQ):
    """ Function to transpose a polynomial matrix. """
    rQ, cQ = polsize(Q, degQ)
    rP = cQ
    cP = rQ
    degP = degQ
    P = pl.zeros((rP, (degP+1)*cP))
    for i in range(degP+1):
        P[:, i*cP:(i+1)*cP] = Q[:, i*cQ:(i+1)*cQ].T

    return P, degP


if __name__== "__main__":

    # print "Test for polsize"
    # print polsize([1, 2, 1],4)

    # print "Test for polmul"
    # C = pl.array([[1, 0, 0.5, 2], [0, 1, -4.71, 2.8]]) 
    # A = pl.array([0.5, 3.5])
    # print polmul(A, 0, C, 1)

    # print "Test for polsplit3"
    # print polsplit3([1, -0.37])

    print "Test for putin"
    A = pl.array([0,0])
    B = pl.array([0.44, -1.6, 1.6, -0.44])
    print putin(A, 0, B, 3, 0, 0)

    
    pass
